# api: streamtuner2
# encoding: utf-8
# title: Shoutcast.com
# description: Primary list of shoutcast servers (now managed by radionomy).
# type: channel
# category: radio
# author: Mario
# original: Jean-Yves Lefort
# version: 1.8
# url: http://directory.shoutcast.com/
# config:
#    { name: shoutcast_format, type: select, select: pls|m3u|xspf|raw, value: pls, description: "Shoutcast playlist format to retrieve" }
#    { name: shoutcast_api, type: bool, value: 0, description: "Use Shoutcast developer API (key from .netrc)" }
# priority: default
# suggests: python:shoutcast-api
# png:
#   iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABHNCSVQICAgIfAhkiAAAAelJREFUOI2NU0toE2EYnM12t2wLkhSXSIgEJMHFQ2naQ+kpoPYQoaXH3gRFsegloUhRQTyU2oOgggQUzzlEQomIBzU+EHooBIol0cOGLqFFFiJ5SB5skvFU6ebRduA7/DAz
#   /PM9BJLoh3Q6zVQqBZfLhXq9jlAohHA4LHTzhvqJ2+02c7kcgsEgfD4fRFFEPp+HZVmUJEk41kAURcHv99Pj8cAwDGiaBkVR0C0GAJDsW7VajYVCgYlEguVymZZlsVKpcG1tlYd5fX8AAIqiCF6vF6VSibIsI5lMYvvDE1xymwDu/ec5BhkcIJPJIHJzFqf372P1cgMf
#   f46cLIKu61yJXufr5VO0voyzEZ/k8sI4s9ns0RFarRZjL56inIshekWGenYS6IzhR9PCntRBIBCw8XsiFItFNLMxPJgfwVjDi4Y8g2b9DILaMKZGd2Ca5tEGiqJg2xjF200H6J+AvKtjeG8T3998xW5nAk6n08bviSBJEqhewLlpN4bMHfwxfuH5J8J98SGerS/B4XDY
#   d+FwQ6rVKm8vXeP++6vku2lu3FEZubFIXdc5qNm2x93ILZobszRfaYwuaIzH4wOFfafwt7CFb59/Y0uYx8rLR1BVtXd1u2AzCMwsQg6cx+O5uWOFBxAGnfNJ8Q/z/DNTtgbnsgAAAABJRU5ErkJggg==
# depends: re, ahttp
# extraction-method: json, regex
#
# Shoutcast is a server software for audio streaming. It automatically spools
# station information on shoutcast.com, which today lists over 85000 radios.
#
# It has been aquired by Radionomy in 2014. Since then significant changes
# took place. The former yellow pages API got deprecated. Streamtuner2 now
# utilizes the AJAX interface for speedy playlist discovery.
#
# Optionally can use the https://pypi.org/project/shoutcast-api/ for
# scanning stations. But that requires a developer API key in .netrc
# via `machine shoutcast.com\n password XYZ`. (UNTESTED, and still using
# standard url lookup handler. Their API is actually meant for setting up
# a shoutcast-like website mirror, unsuitable for desktop apps.)
#


import ahttp
from json import loads as json_decode
import re
from config import *
from channels import *
import channels



# SHOUTcast data module
#
# Former API doc: http://wiki.winamp.com/wiki/SHOUTcast_Radio_Directory_API 
# But neither their Wiki nor Bulletin Board provide concrete information on
# the eligibility of open source desktop apps for an authhash.
#
# Therefore we'll be retrieving stuff from the homepage still. The new
# interface conveniently uses JSON already, so let's use that:
#
#   POST http://www.shoutcast.com/Home/BrowseByGenre {genrename: Pop}
#
# Catmap actually has become redundant.
#
class shoutcast(channels.ChannelPlugin):

    # attrs
    base_url = "http://directory.shoutcast.com/"
    listformat = "pls"
    has_search = False   # may be True now
    
    # categories
    categories = ["Top 500", 'Alternative', ['Adult Alternative', 'Britpop', 'Classic Alternative', 'College', 'Dancepunk', 'Dream Pop', 'Emo', 'Goth', 'Grunge', 'Hardcore', 'Indie Pop', 'Indie Rock', 'Industrial', 'LoFi', 'Modern Rock', 'New Wave', 'Noise Pop', 'Post Punk', 'Power Pop', 'Punk', 'Ska', 'Xtreme'], 'Blues', ['Acoustic Blues','Chicago Blues','Contemporary Blues','Country Blues','Delta Blues','Electric Blues','Cajun and Zydeco'],'Classical', ['Baroque','Chamber','Choral','Classical Period','Early Classical','Impressionist','Modern','Opera','Piano','Romantic','Symphony'],'Country', ['Alt Country','Americana','Bluegrass','Classic Country','Contemporary Bluegrass','Contemporary Country','Honky Tonk','Hot Country Hits','Western'],'Easy Listening', ['Exotica','Light Rock','Lounge','Orchestral Pop','Polka','Space Age Pop'],'Electronic', ['Acid House','Ambient','Big Beat','Breakbeat','Dance','Demo','Disco','Downtempo','Drum and Bass','Electro','Garage','Hard House','House','IDM','Jungle','Progressive','Techno','Trance','Tribal','Trip Hop','Dubstep'],'Folk', ['Alternative Folk','Contemporary Folk','Folk Rock','New Acoustic','Traditional Folk','World Folk','Old Time'],'Themes', ['Adult','Best Of','Chill','Eclectic','Experimental','Female','Heartache','Instrumental','LGBT','Love and Romance','Party Mix','Patriotic','Rainy Day Mix','Reality','Sexy','Shuffle','Travel Mix','Tribute','Trippy','Work Mix'],'Rap', ['Alternative Rap','Dirty South','East Coast Rap','Freestyle','Hip Hop','Gangsta Rap','Mixtapes','Old School','Turntablism','Underground Hip Hop','West Coast Rap'],'Inspirational', ['Christian','Christian Metal','Christian Rap','Christian Rock','Classic Christian','Contemporary Gospel','Gospel','Praise and Worship','Sermons and Services','Southern Gospel','Traditional Gospel'],'International', ['African','Arabic','Asian','Bollywood','Brazilian','Caribbean','Celtic','Chinese','European','Filipino','French','Greek','Hawaiian and Pacific','Hindi','Indian','Japanese','Hebrew','Klezmer','Korean','Mediterranean','Middle Eastern','North American','Russian','Soca','South American','Tamil','Worldbeat','Zouk','German','Turkish','Islamic','Afrikaans','Creole'],'Jazz', ['Acid Jazz','Avant Garde','Big Band','Bop','Classic Jazz','Cool Jazz','Fusion','Hard Bop','Latin Jazz','Smooth Jazz','Swing','Vocal Jazz','World Fusion'],'Latin', ['Bachata','Banda','Bossa Nova','Cumbia','Latin Dance','Latin Pop','Latin Rap and Hip Hop','Latin Rock','Mariachi','Merengue','Ranchera','Reggaeton','Regional Mexican','Salsa','Tango','Tejano','Tropicalia','Flamenco','Samba'],'Metal', ['Black Metal','Classic Metal','Extreme Metal','Grindcore','Hair Metal','Heavy Metal','Metalcore','Power Metal','Progressive Metal','Rap Metal','Death Metal','Thrash Metal'],'New Age', ['Environmental','Ethnic Fusion','Healing','Meditation','Spiritual'],'Decades', ['30s','40s','50s','60s','70s','80s','90s','00s'],'Pop', ['Adult Contemporary','Barbershop','Bubblegum Pop','Dance Pop','Idols','Oldies','JPOP','Soft Rock','Teen Pop','Top 40','World Pop','KPOP'],'R&amp;B and Urban', ['Classic R&amp;B','Contemporary R&amp;B','Doo Wop','Funk','Motown','Neo Soul','Quiet Storm','Soul','Urban Contemporary'],'Reggae', ['Contemporary Reggae','Dancehall','Dub','Pop Reggae','Ragga','Rock Steady','Reggae Roots'],'Rock', ['Adult Album Alternative','British Invasion','Classic Rock','Garage Rock','Glam','Hard Rock','Jam Bands','Piano Rock','Prog Rock','Psychedelic','Rock &amp; Roll','Rockabilly','Singer and Songwriter','Surf','JROCK','Celtic Rock'],'Seasonal and Holiday', ['Anniversary','Birthday','Christmas','Halloween','Hanukkah','Honeymoon','Kwanzaa','Valentine','Wedding','Winter'],'Soundtracks', ['Anime','Kids','Original Score','Showtunes','Video Game Music'],'Talk', ['Comedy','Community','Educational','Government','News','Old Time Radio','Other Talk','Political','Scanner','Spoken Word','Sports','Technology','BlogTalk'],'Misc', [],'Public Radio', ['News','Talk','College','Sports','Weather']]
    catmap = {}
    
    # redefine
    streams = {}

    # API usage
    api_key = None
    api_stations = None
    
    def init2(self, parent):
        self.listformat = conf.shoutcast_format
    
        
    # Extracts the category list from www.shoutcast.com,
    def update_categories(self):
        html = ahttp.get(self.base_url)
        #log.DATA( html )
        self.categories = ["Top 500"]
        
        # Genre list in sidebar
        """
           <li id="genre-3" class="sub-genre " genreid="3" parentgenreid="1">
                <a href="/Genre?name=Britpop" onclick="return loadStationsByGenre('Britpop', 3, 1);">Britpop</a>
           </li>
        """
        rx = re.compile(r"loadStationsByGenre\(  '([^']+)' [,\s]* (\d+) [,\s]* (\d+)  \)", re.X)
        subs = rx.findall(html)

        # group
        current = []
        for (title, id, main) in subs:
            #self.catmap[title] = int(id)
            if not int(main):
                self.categories.append(title)
                current = []
                self.categories.append(current)
            else:
                current.append(title)
        # .categories/.catmap get saved by reload_categories()
        

    # downloads stream list from shoutcast for given category
    def update_streams(self, cat, search=None):
        if conf.get("shoutcast_api"):
            try:
                return self.update_streams_api(cat)
            except Exception as e:
                log.ERR(e)

        # page (always one result set á 500 entries)
        if cat in ("top", "Top", "Top 500"):
            url = self.base_url + "Home/Top"
            params = {}
        elif cat:
            url = self.base_url + "Home/BrowseByGenre"
            params = { "genrename": cat }
        elif search:
            url = self.base_url + "Search/UpdateSearch"
            params = { "query": search }
        referer = self.base_url
        try:
            json = ahttp.get(url, params=params, referer=referer, post=1, ajax=1)
            json = json_decode(json)
        except:
            log.ERR("HTTP request or JSON decoding failed. Outdated python/requests perhaps.")
            return []
        self.parent.status(0.75)

        # remap JSON
        """
            AACEnabled:0
            Bitrate:128
            CurrentTrack:"Bolland & Bolland - Answer For A Lifetime"
            Format:"audio/mpeg"
            Genre:"Folk Rock"
            ID:99180411
            IceUrl:"AOLMRadio"
            IsAACEnabled:false
            IsPlaying:false
            IsRadionomy:true
            Listeners:159
            Name:"AOLMRadio"
            StreamUrl:null
        """
        entries = []
        for e in json:
            if conf.shoutcast_format in ("raw","json","href"):
                url = "urn:shoutcast:{}".format(e.get("ID",0))
            else:
                url = "http://yp.shoutcast.com/sbin/tunein-station.%s?id=%s" % (conf.shoutcast_format, e.get("ID", "0"))
            entries.append({
                "id": int(e.get("ID", 0)),
                "genre": str(e.get("Genre", "")),
                "title": str(e.get("Name", "")),
                "playing": str(e.get("CurrentTrack", "")),
                "bitrate": int(e.get("Bitrate", 0)),
                "listeners": int(e.get("Listeners", 0)),
                "url": url,
                "listformat": conf.shoutcast_format,
                "homepage": "",
                "format": str(e.get("Format", ""))
            })

        #log.DATA(entries)
        return entries
        
    
    # import API module
    def prepare_api(self):
        if not self.api_key:
            self.api_key = conf.netrc(["shoutcast.com"])[2]
        if not self.api_stations:
            from shoutcast_api import stations
            self.api_stations = stations

    # via developer API
    def update_streams_api(self, cat):
        self.prepare_api()
        r = []
        for e in self.api_stations.get_stations_by_genre(self.api_key, cat):
            r.append({
                "title": e["name"],
                "id": e["id"],
                "bitrate": e["br"],
                "playing": e["ct"],
                "img": e["logo_url"],
                "listeners": e["lc"],
                "url": "urn:shoutcast:{}".format(e["id"]),
                "genre": e["genre"],
                "format": "audio/mpeg",
                "listformat": "href",
            })
        return r

    # in case we're using AJAX lookups over tunein-station.pls
    def resolve_urn(self, row):
        if not row.get("id") or not row.get("url", "").startswith("urn:shoutcast:"):
            return
        url = ahttp.get("https://directory.shoutcast.com/Player/GetStreamUrl", {"station":row["id"]}, post=1)
        row["url"] = json_decode(url)  # response is just a string literal of streaming url
        return row["url"]

